<?php

namespace App\Http\Controllers\Api;

use App\Services\OdooClient;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class OdooProductController extends Controller
{
    public function index(Request $request, OdooClient $odoo)
    {
        // Model: template (default) or variant
        $model = $request->get('model', 'product.template'); // or 'product.product'

        // --- Build domain (start with inbound domain if provided) ---
        $domain = json_decode($request->get('domain', '[]'), true) ?: [];

        // --- Multi-category by IDs ---
        // Accept: category_ids=1,2,3 OR category_ids=[1,2,3]
        $categoryIdsParam = $request->get('category_ids');
        $categoryIds = [];
        if ($categoryIdsParam) {
            if (is_string($categoryIdsParam) && str_starts_with(trim($categoryIdsParam), '[')) {
                $categoryIds = json_decode($categoryIdsParam, true) ?: [];
            } else {
                $categoryIds = array_filter(array_map(
                    fn ($v) => (int) trim($v),
                    explode(',', (string) $categoryIdsParam)
                ));
            }
        }

        // --- Multi-category by NAMES ---
        // Accept: categories=Office,Supplies,Furniture OR categories=["Office","Furniture"]
        $categoryNamesParam = $request->get('categories');
        $categoryNames = [];
        if ($categoryNamesParam) {
            if (is_string($categoryNamesParam) && str_starts_with(trim($categoryNamesParam), '[')) {
                $categoryNames = json_decode($categoryNamesParam, true) ?: [];
            } else {
                $categoryNames = array_filter(array_map('trim', explode(',', (string) $categoryNamesParam)));
            }
        }

        // If we have categories, choose IN or CHILD_OF per flag
        $includeChildren = $request->boolean('include_children', false);
        if (!empty($categoryIds)) {
            if ($includeChildren) {
                // all descendants of any given category id
                $domain[] = ['categ_id', 'child_of', $categoryIds];
            } else {
                $domain[] = ['categ_id', 'in', $categoryIds];
            }
        }

        // --- Fields / pagination ---
        $fields = $request->has('fields')
            ? array_values(array_filter(array_map('trim', explode(',', $request->get('fields')))))
            : ['id','name','list_price','standard_price','type','uom_id','categ_id','default_code','barcode','active'];

        $limit  = (int) $request->get('limit', 100);
        $offset = (int) $request->get('offset', 0);

        // --- Fetch ---
        if (!$request->boolean('all', false)) {
            $rows = $odoo->getProducts($model, $domain, $fields, $limit, $offset, [
                // 'order'   => 'name asc',
                // 'context' => ['lang' => 'en_US'],
            ]);
            return response()->json(['count' => count($rows), 'items' => $rows]);
        }

        $rows = $odoo->getAllProducts($model, $domain, $fields, 200, [
            // 'order'   => 'name asc',
            // 'context' => ['lang' => 'en_US'],
        ]);

        return response()->json(['count' => count($rows), 'items' => $rows]);
    }

    public function createOrder(Request $request, OdooClient $odoo)
    {
        $request->validate([
            'pos_config_id' => 'required|integer',
            'items'         => 'required|array|min:1',
            'items.*.product_id' => 'required|integer',
            'items.*.qty'        => 'required|numeric',
            'items.*.price_unit' => 'required|numeric',
            'amount_total'       => 'nullable|numeric',
        ]);

        // 1. Get/opened session
        $sessionId = $request->integer('session_id') ?: $odoo->getOpenPosSessionId(
            $request->integer('pos_config_id')
        );

        if (!$sessionId) {
            return response()->json([
                'status'  => 'error',
                'message' => 'No opened POS session found for given pos_config_id.',
            ], 422);
        }

        // 2. Build one2many "lines" commands for pos.order.line
        // Format: [ [0, 0, {vals}], [0, 0, {vals}], ... ]
        $lines = collect($request->input('items'))->map(function ($item) {
            return [
                0,
                0,
                [
                    'product_id' => (int) $item['product_id'],
                    'qty'        => (float) $item['qty'],
                    'price_unit' => (float) $item['price_unit'],
                    // Optional:
                    // 'discount'  => 0,
                    // 'tax_ids'   => [[6, 0, [tax_id1, tax_id2]]],
                ],
            ];
        })->toArray();

        // 3. Build order values (pos.order)
        $orderValues = [
            'session_id'   => $sessionId,
            'partner_id'   => $request->integer('customer_id') ?: null,
            'lines'        => $lines,
            // You can send 'amount_total' or let Odoo compute it
            'amount_total' => $request->input('amount_total'),
            // Optional more fields:
            // 'pricelist_id' => ...,
            // 'note'         => $request->input('note'),
        ];

        try {
            $orderId = $odoo->createPosOrder($orderValues);

            return response()->json([
                'status'           => 'ok',
                'odoo_pos_order_id'=> $orderId,
                'order_values'     => $orderValues,
            ]);
        } catch (\Throwable $e) {
            return response()->json([
                'status'  => 'error',
                'message' => $e->getMessage(),
            ], 500);
        }
    }

}
