<?php

namespace App\Http\Controllers\Api;

use App\Services\OdooClient;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class OdooProductController extends Controller
{
    public function index(Request $request, OdooClient $odoo)
    {
        // Model: template (default) or variant
        $model = $request->get('model', 'product.product'); // or 'product.product'

        // --- Build domain (start with inbound domain if provided) ---
        $domain = json_decode($request->get('domain', '[]'), true) ?: [];

        // --- Multi-category by IDs ---
        $categoryIdsParam = $request->get('category_ids');
        $categoryIds = [];
        if ($categoryIdsParam) {
            if (is_string($categoryIdsParam) && str_starts_with(trim($categoryIdsParam), '[')) {
                $categoryIds = json_decode($categoryIdsParam, true) ?: [];
            } else {
                $categoryIds = array_filter(array_map(
                    fn ($v) => (int) trim($v),
                    explode(',', (string) $categoryIdsParam)
                ));
            }
        }

        // --- Multi-category by NAMES ---
        $categoryNamesParam = $request->get('categories');
        $categoryNames = [];
        if ($categoryNamesParam) {
            if (is_string($categoryNamesParam) && str_starts_with(trim($categoryNamesParam), '[')) {
                $categoryNames = json_decode($categoryNamesParam, true) ?: [];
            } else {
                $categoryNames = array_filter(array_map('trim', explode(',', (string) $categoryNamesParam)));
            }
        }

        // Resolve category NAMES -> IDs (optional)
        if (!empty($categoryNames)) {
            $cats = $odoo->callKw('product.category', 'search_read', [
                [['name', 'in', $categoryNames]]
            ], [
                'fields' => ['id', 'name'],
                'limit'  => 0,
            ]);

            $nameCatIds = array_values(array_filter(array_map(fn($c) => $c['id'] ?? null, $cats)));
            $categoryIds = array_values(array_unique(array_merge($categoryIds, $nameCatIds)));
        }

        // If we have categories, choose IN or CHILD_OF per flag
        $includeChildren = $request->boolean('include_children', false);
        if (!empty($categoryIds)) {
            $domain[] = $includeChildren
                ? ['categ_id', 'child_of', $categoryIds]
                : ['categ_id', 'in', $categoryIds];
        }

        // --- Fields / pagination ---
        $fields = $request->has('fields')
            ? array_values(array_filter(array_map('trim', explode(',', $request->get('fields')))))
            : [
                'id','name','list_price','standard_price','type',
                'uom_id','categ_id','default_code','barcode','active'
                // 'product_variant_ids','product_variant_count'
                ,'attribute_line_ids'
            ];

        $limit  = (int) $request->get('limit', 100);
        $offset = (int) $request->get('offset', 0);

        // --- Fetch ---
        if (!$request->boolean('all', false)) {
            $rows = $odoo->getProducts($model, $domain, $fields, $limit, $offset);
        } else {
            $rows = $odoo->getAllProducts($model, $domain, $fields, 200);
        }

        // ✅ Expand relation names (default true)
        $expand = $request->boolean('expand_relations', true);

        if ($expand && !empty($rows)) {
            // 1) Collect IDs
            $variantIds = [];
            $lineIds = [];

            foreach ($rows as $r) {
                if (!empty($r['product_variant_ids'])) $variantIds = array_merge($variantIds, $r['product_variant_ids']);
                if (!empty($r['attribute_line_ids']))  $lineIds    = array_merge($lineIds, $r['attribute_line_ids']);
            }

            $variantIds = array_values(array_unique(array_filter($variantIds)));
            $lineIds    = array_values(array_unique(array_filter($lineIds)));

            // 2) Variant ID -> display_name
            $variantMap = [];
            if (!empty($variantIds)) {
                $variantRows = $odoo->callKw('product.product', 'search_read', [
                    [['id', 'in', $variantIds]]
                ], [
                    'fields' => ['id', 'display_name'],
                    'limit'  => 0,
                ]);

                foreach ($variantRows as $vr) {
                    $variantMap[$vr['id']] = $vr['display_name'] ?? null;
                }
            }

            // 3) Attribute line ID -> {attribute, values[]}
            $attributeLineMap = [];
            if (!empty($lineIds)) {
                $lineRows = $odoo->callKw('product.template.attribute.line', 'search_read', [
                    [['id', 'in', $lineIds]]
                ], [
                    'fields' => ['id', 'attribute_id', 'value_ids'],
                    'limit'  => 0,
                ]);

                // collect all value_ids
                $valueIds = [];
                foreach ($lineRows as $lr) {
                    if (!empty($lr['value_ids'])) $valueIds = array_merge($valueIds, $lr['value_ids']);
                }
                $valueIds = array_values(array_unique(array_filter($valueIds)));

                // value id -> name
                $valueMap = [];
                if (!empty($valueIds)) {
                    $valueRows = $odoo->callKw('product.attribute.value', 'search_read', [
                        [['id', 'in', $valueIds]]
                    ], [
                        'fields' => ['id', 'name'],
                        'limit'  => 0,
                    ]);

                    foreach ($valueRows as $vv) {
                        $valueMap[$vv['id']] = $vv['name'] ?? null;
                    }
                }

                // build attribute line map
                foreach ($lineRows as $lr) {
                    $attrName = null;

                    if (!empty($lr['attribute_id']) && is_array($lr['attribute_id'])) {
                        $attrName = $lr['attribute_id'][1] ?? null; // [id, name]
                    }

                    $values = [];

                    foreach (($lr['value_ids'] ?? []) as $vid) {
                        if (isset($valueMap[$vid])) {
                            $values[] = [
                                'id'   => $vid,
                                'name' => $valueMap[$vid],
                            ];
                        }
                    }

                    $attributeLineMap[$lr['id']] = [
                        'attribute' => $attrName,
                        'values'    => $values,
                    ];
                }
            }

            // 4) Replace fields per product row
            foreach ($rows as &$r) {
                $r['product_variants'] = array_values(array_filter(array_map(
                    fn ($id) => $variantMap[$id] ?? null,
                    $r['product_variant_ids'] ?? []
                )));

                $r['attributes'] = array_values(array_filter(array_map(
                    fn ($id) => $attributeLineMap[$id] ?? null,
                    $r['attribute_line_ids'] ?? []
                )));

                unset($r['product_variant_ids'], $r['attribute_line_ids']);
            }
            unset($r);
        }

        return response()->json([
            'count' => count($rows),
            'items' => $rows,
        ]);
    }
    public function createOrder(Request $request, OdooClient $odoo)
    {
        // return 1;
        // $request->validate([
        //     'pos_config_id' => 'required|integer',
        //     'items'         => 'required|array|min:1',
        //     'items.*.product_id' => 'required|integer',
        //     'items.*.qty'        => 'required|numeric',
        //     'items.*.price_unit' => 'required|numeric',
        //     'amount_total'       => 'nullable|numeric',
        // ]);

        // 1. Get/opened session
        $sessionId = $odoo->getOpenPosSessionId(3);

        // if (!$sessionId) {
        //     return response()->json([
        //         'status'  => 'error',
        //         'message' => 'No opened POS session found for given pos_config_id.',
        //     ], 422);
        // }

        // 2. Build one2many "lines" commands for pos.order.line
        // Format: [ [0, 0, {vals}], [0, 0, {vals}], ... ]
        // $lines = collect($request->input('items'))->map(function ($item) {
        //     return [
        //         0,
        //         0,
        //         [
        //             'product_id' => 2,
        //             'qty'        => (float) 1,
        //             'price_unit' => (float) 1,
        //             // Optional:
        //             // 'discount'  => 0,
        //             // 'tax_ids'   => [[6, 0, [tax_id1, tax_id2]]],
        //         ],
        //     ];
        // })->toArray();
        $qty        = 1.0;
        $priceUnit  = 1.0;
        $lineSubtot = $qty * $priceUnit;   // 1.0 (tax excl)
        $lineTotal  = $lineSubtot;   
        $attrText = "Part: Inner, Size: Small, Level: Beginner, Color: Black";
        
        $lines[] = 
         [
            0,
            0,
            [
                'product_id'           => 5266,
                'qty'                  => $qty,
                'price_unit'           => $priceUnit,
                'customer_note'        => $attrText,
                'tax_ids'              => [],          // no tax
                'price_subtotal'       => $lineSubtot, // ðŸ”´ REQUIRED
                'price_subtotal_incl'  => $lineTotal,  // good to send too
            ],
        ];
       
        $subtotal   = $lineSubtot;  // 1.0
        $taxAmount  = 0.0;
        $total      = $subtotal + $taxAmount;
        $amountPaid = $total;
        $change     = 0.0;
        // 3. Build order values (pos.order)
        $orderValues = [
            'session_id'    => $sessionId,
            'partner_id'    => $request->integer('customer_id') ?: false,
        
            'lines'         => $lines,
        
            // ðŸ”´ Header monetary fields â€“ all numeric
            'amount_tax'    => $taxAmount,   // 0.0
            'amount_total'  => $total,       // 1.0
            'amount_paid'   => $amountPaid,  // 1.0
            'amount_return' => $change,      // 0.0
        ];

        try {
            $orderId = $odoo->createPosOrder($orderValues);

            return response()->json([
                'status'           => 'ok',
                'odoo_pos_order_id'=> $orderId,
                'order_values'     => $orderValues,
            ]);
        } catch (\Throwable $e) {
            return response()->json([
                'status'  => 'error',
                'message' => $e->getMessage(),
            ], 500);
        }
    }

}
